/**
 * \file sdc_op_encrypt_decrypt .c
 *
 * \brief Functions for encrypt and decrypt
 * Please note : the implementation is split into different operations
 * instead of splitting it in common, convenience and advanced functions
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <string.h>
#include <sdc_op_common.h>
#include <sdc_op_adv.h>
#include <sdc_op_conv.h>
#include <sdc_random.h>
#include <private/sdc_arch.h>
#include <private/sdc_intern.h>

/* Definitions types and defaults */

static const char *sdc_encrypt_decrypt_alg_names[] = {
    [SDC_ENCDEC_ALG_AES] = "AES",
    [SDC_ENCDEC_ALG_DES] = "DES",
    [SDC_ENCDEC_ALG_3DES] = "3DES",
    [SDC_ENCDEC_ALG_RSA] = "RSA"
};

static const char *sdc_encrypt_decrypt_blk_names[] = {
    [SDC_ENCDEC_BLK_ECB] = "ECB",
    [SDC_ENCDEC_BLK_CTR] = "CTR",
    [SDC_ENCDEC_BLK_CBC] = "CBC",
    [SDC_ENCDEC_BLK_CFB] = "CFB",
    [SDC_ENCDEC_BLK_OFB] = "OFB",
    [SDC_ENCDEC_BLK_NONE] = NULL
};

/* Functions */

/**
 * \brief Checks common to encrypt and decrypt
 *
 * Check session, type and in and out data
 * Provide \ref sdc_encrypt_decrypt_desc_t of type
 */
static sdc_error_t sdc_encrypt_decrypt_common_checks_defaults(
    sdc_error_t error_init,
    sdc_session_t *session,
    const sdc_encrypt_decrypt_type_t *type,
    sdc_encrypt_decrypt_desc_t *internal_desc,
    const uint8_t *in_data, const size_t in_len,
    uint8_t **out_data, size_t *out_len, const sdc_error_t out_err)
{

    sdc_error_t err = error_init;

    if (SDC_OK != sdc_intern_check_init_data_output_buffer(out_data, out_len))
        err = out_err;

    if (SDC_OK != sdc_intern_check_data_input_buffer(in_data, in_len))
        err = SDC_IN_DATA_INVALID;

    if (!type)
        err = SDC_ALG_MODE_INVALID;

    if (!session)
        err = SDC_SESSION_INVALID;

    if (err == SDC_OK)
        err = sdc_encrypt_decrypt_desc_fill (session, type, internal_desc);

    return err;
}

/**
 * \brief call min max checks for in data and iv
 */
static sdc_error_t sdc_encrypt_decrypt_common_check_iv_in_length(
    const sdc_encrypt_decrypt_desc_t *desc,
    bool is_encrypt,
    size_t in_len,
    size_t iv_len
    )
{
    sdc_error_t err;

    err = sdc_intern_inout_input_check_min_max_align(in_len,
                                                           &(desc->data),
                                                           /* use plain when encrypt else cipher spec */
                                                           is_encrypt);
    if (err != SDC_OK)
        return err;

    if (SDC_OK != sdc_intern_range_min_max_mod_check(iv_len, &(desc->iv)))
        return SDC_IV_INVALID;

    return SDC_OK;
}

static sdc_error_t sdc_common_encdec_init(sdc_session_t *session,
                                  const sdc_encrypt_decrypt_type_t *type,
                                  const sdc_encrypt_decrypt_desc_t *desc,
                                  const uint8_t *iv, size_t iv_len,
                                  bool is_encrypt)
{
    session->unaligned_buffer_filllevel = 0;
    session->inlen_cnt = 0;

    if (!desc->supports_iuf)
        return SDC_OP_NOT_SUPPORTED;

    if (is_encrypt)
        return sdc_arch_encrypt_init(session, type, desc, iv, iv_len);

    return sdc_arch_decrypt_init(session, type, desc, iv, iv_len);
}

static sdc_error_t sdc_common_encdec_update(sdc_session_t *session,
                                             const sdc_encrypt_decrypt_type_t *type,
                                             const sdc_encrypt_decrypt_desc_t *desc,
                                             const uint8_t *in_data, const size_t in_data_len,
                                             uint8_t *out_data, size_t *out_data_len,
                                             bool is_encrypt)
{
    sdc_error_t err = SDC_OK;
    sdc_padding_t padding = desc->data.padding;
    size_t block_len = desc->data.block_len;
    size_t chunk_len = desc->data.max_chunk_len;
    bool chunk_len_aligned = desc->data.chunk_len_aligned;
    const uint8_t *next_in_data;
    size_t in_len_remaining;
    uint8_t *next_out_data;
    size_t out_len_remaining;
    size_t out_len_used;
    size_t unaligned_buf_rem;
    uint8_t *unaligned_buf_next;
    size_t used_out_len;
    size_t min_no_proc_len;
    size_t process_in_len;
    size_t remainder;
    bool hold_back = false;

    if ((!is_encrypt) &&
        (padding != SDC_PADDING_INTERNAL) &&
        (padding != SDC_PADDING_NO) &&
        (padding != SDC_PADDING_HIDDEN))
    {
        /*
         * In order to perform unpad we need some data in finalize
         * One easy way is to always keep some data left in
         * the unaligned buffer
         *
         * The downside of this approach is that each (but the first)
         * SW update call will result in to calls two arch and two ioctls
         *
         * Note: This mode is also reflected in sdc_decrypt_get_update_len
         */
        hold_back = true;
    }

    if(in_data_len > *out_data_len)
        err = SDC_OUT_DATA_INVALID;

    if (session->unaligned_buffer_filllevel > block_len)
        err = SDC_INTERNAL_ERROR;

    /* Check that processing additional data won't exceed max */
    if (err == SDC_OK)
        err = sdc_intern_inout_input_check_update_wont_exceed_max(
                session->inlen_cnt,
                in_data_len,
                &(desc->data),
                /* use plain when encrypt else cipher spec */
                is_encrypt);

    next_in_data = in_data;
    in_len_remaining = in_data_len;
    next_out_data = out_data;
    out_len_remaining = *out_data_len;
    out_len_used = 0;

    /* handle previous unaligned data first */
    if ((err == SDC_OK) && (session->unaligned_buffer_filllevel > 0)) {
        /* if unaligned_buffer_filllevel > 0 we obviously need to alignment */
        unaligned_buf_rem = block_len - session->unaligned_buffer_filllevel;

        if (!chunk_len_aligned) {
            /*
             * This must not happen
             * Probably someone has corrupted the control structures
             */
            err = SDC_INTERNAL_ERROR;
        } else {
            if ((in_len_remaining > unaligned_buf_rem) ||
                ((in_len_remaining >= unaligned_buf_rem) && (!hold_back)))
            {
                unaligned_buf_next = session->unaligned_buffer;
                unaligned_buf_next += session->unaligned_buffer_filllevel;

                /* afterwards we have on block in the buffer */
                memcpy(unaligned_buf_next,
                       next_in_data,
                       unaligned_buf_rem);

                /* process the unaligned part */
                used_out_len = out_len_remaining;
                if (is_encrypt) {
                    err = sdc_arch_encrypt_update(session, type, desc,
                                                  session->unaligned_buffer, block_len,
                                                  next_out_data, &used_out_len);
                } else {
                    err = sdc_arch_decrypt_update(session, type, desc,
                                                  session->unaligned_buffer, block_len,
                                                  next_out_data, &used_out_len);
                }
                if (err == SDC_OK) {
                    /* update pointers + lengths for remaining data */
                    next_in_data += unaligned_buf_rem;
                    in_len_remaining -= unaligned_buf_rem;
                    next_out_data += used_out_len;
                    out_len_remaining -= used_out_len;
                    out_len_used += used_out_len;

                    /* all data handled */
                    session->unaligned_buffer_filllevel = 0;
                }
            }
            /*
             * else case
             *       i.e. in_len_remaining < unaligned_buf_rem (hold_back == false)
             *       i.e. in_len_remaining <= unaligned_buf_rem (hold_back == true)
             * will be handled by "append remaining data to unaligned_buffer"
             *
             * In this case
             *      in_len_remaining is < block_len (hold_back == false)
             *      in_len_remaining is <= block_len (hold_back == true)
             * as well and chunk_len_aligned or hold_back needs to be true.
             *
             * Otherwise there is no way (beside fooling around with internal
             * structs) that data has been added to unaligned_buffer in the
             * previous call.
             *
             * Note: the while loop won't process any data in this case
             */
        }
    }

    /* smaller or equal length need to be handled using unaligned_buffer */
    min_no_proc_len = 0;
    if (chunk_len_aligned)
        min_no_proc_len = block_len - 1;
    if (hold_back)
        min_no_proc_len = block_len;

    while ((err == SDC_OK) && (in_len_remaining > min_no_proc_len)) {
        process_in_len = in_len_remaining;

        if (process_in_len > chunk_len)
            process_in_len = chunk_len;

        if (chunk_len_aligned) {
            /* align */
            remainder = (process_in_len % block_len);
            if (remainder != 0) {
                process_in_len -= remainder;
            }
        }

        if ((hold_back) && (process_in_len == in_len_remaining)) {
            /*
             * hold back some data
             * in this case process_in_len >= block_len + 1
             * Note: due to min_no_proc_len : process_in_len == in_len_remaining > block_len
             */
            process_in_len -= block_len;
        }

        /* process the next chunk of data */
        used_out_len = out_len_remaining;
        if (is_encrypt) {
            err = sdc_arch_encrypt_update(session, type, desc,
                                          next_in_data, process_in_len,
                                          next_out_data, &used_out_len);
        } else {
            err = sdc_arch_decrypt_update(session, type, desc,
                                          next_in_data, process_in_len,
                                          next_out_data, &used_out_len);
        }

        if (err == SDC_OK) {
            /* update pointers + lengths for remaining data */
            next_in_data += process_in_len;
            in_len_remaining -= process_in_len;
            next_out_data += used_out_len;
            out_len_remaining -= used_out_len;
            out_len_used += used_out_len;
        }
    }

    /* append remaining data to unaligned_buffer */
    if ((err == SDC_OK) && (in_len_remaining > 0)) {
        unaligned_buf_rem = block_len - session->unaligned_buffer_filllevel;

        if (in_len_remaining > unaligned_buf_rem) {
            /* this must not happen */
            err = SDC_INTERNAL_ERROR;
        } else {
            unaligned_buf_next = session->unaligned_buffer;
            unaligned_buf_next += session->unaligned_buffer_filllevel;

            /* append to the end of the unaligned buffer */
            memcpy(unaligned_buf_next,
                   next_in_data,
                   in_len_remaining);
            session->unaligned_buffer_filllevel += in_len_remaining;

            /* no need to update in_data_remaining */
            in_len_remaining = 0;
        }
    }

    *out_data_len = out_len_used;

    if (err != SDC_OK) {
        /* clear confidential data in case of error */
        sdc_intern_clear_session_confidential(session);
    } else {
        /* add the current input data length */
        session->inlen_cnt += in_data_len;
    }

    return err;
}

static sdc_error_t sdc_common_encdec_finalize(sdc_session_t *session,
                                             const sdc_encrypt_decrypt_type_t *type,
                                             const sdc_encrypt_decrypt_desc_t *desc,
                                             uint8_t *out_data, size_t *out_data_len,
                                             bool is_encrypt)
{
    sdc_error_t err = SDC_OK;
    size_t unaligned_data_len;
    sdc_padding_t padding = desc->data.padding;
    size_t block_len = desc->data.block_len;
    bool padded = false;

    unaligned_data_len = session->unaligned_buffer_filllevel;

    if ((padding != SDC_PADDING_INTERNAL) &&
        (padding != SDC_PADDING_NO) &&
        (padding != SDC_PADDING_HIDDEN))
        padded = true;

    err = sdc_intern_inout_input_check_min_max_align(
            session->inlen_cnt,
            &(desc->data),
            /* use plain when encrypt else cipher spec */
            is_encrypt);

    if (err == SDC_OK) {
        if (is_encrypt) {
            if (padded) {
                err = sdc_intern_pad(padding,
                                     session->unaligned_buffer,
                                     unaligned_data_len,
                                     block_len);
                unaligned_data_len = block_len;
            }

            if (err == SDC_OK)
                err = sdc_arch_encrypt_finalize(session, type, desc,
                                                session->unaligned_buffer,
                                                unaligned_data_len,
                                                out_data, out_data_len);
        } else {
            if (padded) {
                /* padded data needs to be aligned to the block length */
                if ((unaligned_data_len % block_len) != 0)
                    err = SDC_IN_DATA_INVALID;
            }

            if (err == SDC_OK)
                err = sdc_arch_decrypt_finalize(session, type, desc,
                                                session->unaligned_buffer,
                                                unaligned_data_len,
                                                out_data, out_data_len);

            if ((err == SDC_OK) && (padding != SDC_PADDING_INTERNAL))
                err = sdc_intern_unpad(padding,
                                       out_data,
                                       out_data_len,
                                       block_len);
        }
    }

    sdc_intern_clear_session_confidential(session);

    return err;
}

/**
 * \brief common implementation of encrypt using architecture dependent
 * init, update, finalize functions
 *
 * This function will be used when the architecture dependent part does not provide
 * optimized encrypt functionality
 *
 */
static sdc_error_t sdc_common_encrypt(sdc_session_t *session,
                                      const sdc_encrypt_decrypt_type_t *type,
                                      const sdc_encrypt_decrypt_desc_t *desc,
                                      const uint8_t *in_data,
                                      const size_t in_data_len,
                                      uint8_t *out_data,
                                      const size_t out_data_len,
                                      const uint8_t *iv, const size_t iv_len)
{
    sdc_error_t err = SDC_OK;
    size_t used_out_data;
    uint8_t *next_out_data = out_data;
    size_t remaining_out_data_len = out_data_len;

    err = sdc_common_encdec_init(session, type, desc, iv, iv_len, true);

    if (err == SDC_OK) {
        used_out_data = remaining_out_data_len;
        err = sdc_common_encdec_update(session, type, desc,
                                       in_data, in_data_len,
                                       next_out_data, &used_out_data,
                                       true);
        next_out_data += used_out_data;
        remaining_out_data_len -= used_out_data;
    }

    if (err == SDC_OK) {
        used_out_data = remaining_out_data_len;
        err = sdc_common_encdec_finalize(session, type, desc,
                                         next_out_data, &used_out_data,
                                         true);
        /* not needed: next_out_data += used_out_data; */
        remaining_out_data_len -= used_out_data;
    }

    /*
     * for encrypt the exact output length is known in advance
     * therefore the complete buffer needs to be used
     */
    if ((err == SDC_OK) && (remaining_out_data_len != 0))
        err = SDC_INTERNAL_ERROR;

    return err;
}

/**
 * \brief common implementation of decrypt using architecture dependent
 * init, update, finalize functions
 *
 * This function will be used when the architecture dependent part does not provide
 * optimized decrypt functionality
 *
 */
static sdc_error_t sdc_common_decrypt(sdc_session_t *session,
                                      const sdc_encrypt_decrypt_type_t *type,
                                      const sdc_encrypt_decrypt_desc_t *desc,
                                      const uint8_t *in_data,
                                      const size_t in_data_len,
                                      uint8_t *out_data,
                                      size_t *out_data_len,
                                      const uint8_t *iv, const size_t iv_len)
{
    sdc_error_t err = SDC_OK;
    size_t used_out_data;
    uint8_t *next_out_data = out_data;
    size_t remaining_out_data_len = *out_data_len;
    size_t sum_out_len = 0;

    err = sdc_common_encdec_init(session, type, desc, iv, iv_len, false);

    if (err == SDC_OK) {
        used_out_data = remaining_out_data_len;
        err = sdc_common_encdec_update(session, type, desc,
                                       in_data, in_data_len,
                                       next_out_data, &used_out_data,
                                       false);
        next_out_data += used_out_data;
        remaining_out_data_len -= used_out_data;
        sum_out_len += used_out_data;
    }

    if (err == SDC_OK) {
        used_out_data = remaining_out_data_len;
        err = sdc_common_encdec_finalize(session, type, desc,
                                         next_out_data, &used_out_data,
                                         false);
        /* not needed: next_out_data += used_out_data; */
        /* not needed: remaining_out_data_len -= used_out_data; */
        sum_out_len += used_out_data;
    }

    if (err == SDC_OK)
        *out_data_len = sum_out_len;

    return err;
}

/**
 * \brief select if architecture specific or common version is used
 */
static sdc_error_t sdc_encrypt_selector(sdc_session_t *session,
                                        const sdc_encrypt_decrypt_type_t *type,
                                        const sdc_encrypt_decrypt_desc_t *desc,
                                        const uint8_t *in_data,
                                        const size_t in_data_len,
                                        uint8_t *out_data,
                                        const size_t out_data_len,
                                        const uint8_t *iv, const size_t iv_len)
{

    sdc_error_t err;

    err = sdc_arch_encrypt(session,
                           type,
                           desc,
                           in_data,
                           in_data_len,
                           out_data,
                           out_data_len,
                           iv, iv_len);

    if (err == SDC_NOT_SUPPORTED) {
        err = sdc_common_encrypt(session,
                                 type,
                                 desc,
                                 in_data,
                                 in_data_len,
                                 out_data,
                                 out_data_len,
                                 iv, iv_len);
    }

    return err;
}

/**
 * \brief select if architecture specific or common version is used
 */
static sdc_error_t sdc_decrypt_selector(sdc_session_t *session,
                                        const sdc_encrypt_decrypt_type_t *type,
                                        const sdc_encrypt_decrypt_desc_t *desc,
                                        const uint8_t *in_data,
                                        const size_t in_data_len,
                                        uint8_t *out_data,
                                        size_t *out_data_len,
                                        const uint8_t *iv, const size_t iv_len)
{

    sdc_error_t err;

    err = sdc_arch_decrypt(session,
                           type,
                           desc,
                           in_data,
                           in_data_len,
                           out_data,
                           out_data_len,
                           iv, iv_len);

    if (err == SDC_NOT_SUPPORTED) {
        err = sdc_common_decrypt(session,
                                 type,
                                 desc,
                                 in_data,
                                 in_data_len,
                                 out_data,
                                 out_data_len,
                                 iv, iv_len);
    }

    return err;
}


sdc_error_t sdc_encrypt(sdc_session_t *session,
                        const sdc_encrypt_decrypt_type_t *type,
                        const uint8_t *in_data, const size_t in_len,
                        uint8_t **iv, size_t *iv_len,
                        uint8_t **out_data, size_t *out_len)
{

    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_encrypt_decrypt_desc_t internal_desc;

    bool explicit_iv;
    uint8_t *internal_iv;
    size_t internal_iv_len;
    uint8_t *internal_out = NULL;
    size_t internal_out_len = 0;


    /* initialize - in case failing with error we want to return
     * buffer pointer NULL and buffer len 0 for all not explicitly specified buffers
     */
    if (SDC_OK != sdc_intern_check_init_dgst_tag_iv_output_buffer(iv, iv_len, &internal_iv_len, SDC_IV_USE_DEFAULT, &explicit_iv)) {
        err = SDC_IV_INVALID;
    }

    err = sdc_encrypt_decrypt_common_checks_defaults(err,
                                                     session,
                                                     type, &internal_desc,
                                                     in_data, in_len,
                                                     out_data, out_len, SDC_OUT_DATA_INVALID);

    if (err != SDC_OK)
        return err;

    if (explicit_iv) {
        internal_iv = *iv;
    } else {
        if (internal_iv_len == SDC_IV_USE_DEFAULT) {
            internal_iv_len = internal_desc.iv.dflt;
        }
        internal_iv = NULL;
        /* there might be formats without IV */
        if (internal_iv_len != 0) {
            err = sdc_random_gen_buffer(session, internal_iv_len, &internal_iv);
        }
    }

    if (err == SDC_OK) {
        err = sdc_encrypt_decrypt_common_check_iv_in_length(&internal_desc,
                                                            true,
                                                            in_len,
                                                            internal_iv_len);
    }

    if (err == SDC_OK) {
        /* determine the output length */
        err = sdc_arch_encrypt_get_out_len(session, type, &internal_desc, in_len, &internal_out_len);
    }

    if (err == SDC_OK) {
        if (internal_out_len) {
            internal_out = malloc(internal_out_len);
            if (!internal_out)
                err = SDC_NO_MEM;
        }
    }

    if (err == SDC_OK) {
        err = sdc_encrypt_selector(session,
                                   type,
                                   &internal_desc,
                                   in_data,
                                   in_len,
                                   internal_out,
                                   internal_out_len,
                                   internal_iv, internal_iv_len);
    }

    if (err == SDC_OK) {
        if (!explicit_iv) {
            *iv = internal_iv;
            *iv_len = internal_iv_len;
        }
        *out_data = internal_out;
        *out_len = internal_out_len;
    } else {
        /* clean allocated memory */
        if (!explicit_iv)
            free (internal_iv);
        free(internal_out);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_ENCRYPT, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_decrypt(sdc_session_t *session,
                        const sdc_encrypt_decrypt_type_t *type,
                        const uint8_t *in_data, const size_t in_len,
                        const uint8_t *iv, const size_t iv_len,
                        uint8_t **out_data, size_t *out_len)
{

    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;
    sdc_encrypt_decrypt_desc_t internal_desc;
    uint8_t *internal_out = NULL;
    size_t internal_out_len = 0;

    if (SDC_OK != sdc_intern_check_tag_iv_input_buffer(iv, iv_len, SDC_IV_USE_DEFAULT))
        err = SDC_IV_INVALID;

    err = sdc_encrypt_decrypt_common_checks_defaults(err,
                                                     session,
                                                     type, &internal_desc,
                                                     in_data, in_len,
                                                     out_data, out_len, SDC_OUT_DATA_INVALID);

    if (err != SDC_OK)
        return err;

    err = sdc_encrypt_decrypt_common_check_iv_in_length(&internal_desc,
                                                        false,
                                                        in_len,
                                                        iv_len);

    if (err == SDC_OK) {
        /* determine the max output length */
        err = sdc_arch_decrypt_get_max_out_len(
            session, type, &internal_desc,
            in_len,
            &internal_out_len);
    }

    if ((err == SDC_OK) && (internal_out_len != 0)) {
        /* allocate output buffer */
        internal_out = malloc(internal_out_len);
        if (!internal_out)
            err = SDC_NO_MEM;
    }

    if (err == SDC_OK) {
        err = sdc_decrypt_selector(session,
                                   type,
                                   &internal_desc,
                                   in_data,
                                   in_len,
                                   internal_out,
                                   &internal_out_len,
                                   iv, iv_len);
    }

    if (err == SDC_OK) {
        *out_data = internal_out;
        *out_len = internal_out_len;
    } else {
        if (internal_out)
            free(internal_out);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_DECRYPT, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}


sdc_error_t sdc_encrypt_formatted_extended(sdc_session_t *session,
                                           const sdc_encrypt_decrypt_type_t *type,
                                           const uint8_t *in_data, const size_t in_len,
                                           const uint8_t *iv, const size_t iv_len,
                                           uint8_t **formatted_data, size_t *formatted_len)
{

    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;
    sdc_encrypt_decrypt_desc_t internal_desc;
    size_t internal_iv_len;
    bool explicit_iv;
    sdc_form_header_generic_t form_header;
    sdc_form_header_encrypt_decrypt_generic_t *form_encdec;
    uint8_t *internal_formatted_buffer = NULL;
    uint8_t *iv_tmp;
    size_t internal_out_len = 0;

    /* if the iv is externally determined the len has to be set externally too */
    explicit_iv = (iv != NULL);
    if (explicit_iv && (iv_len == SDC_IV_USE_DEFAULT))
        err = SDC_IV_INVALID;

    err = sdc_encrypt_decrypt_common_checks_defaults(err,
                                                     session,
                                                     type, &internal_desc,
                                                     in_data, in_len,
                                                     formatted_data, formatted_len, SDC_FORMATTED_DATA_INVALID);

    if (err != SDC_OK)
        return err;

    /* get defaults if not set by application */
    internal_iv_len = iv_len;
    if (internal_iv_len == SDC_IV_USE_DEFAULT) {
        internal_iv_len = internal_desc.iv.dflt;
    }

    err = sdc_encrypt_decrypt_common_check_iv_in_length(&internal_desc,
                                                        true,
                                                        in_len,
                                                        internal_iv_len);

    if (err == SDC_OK) {
        uint64_t opt_bmsk;

        err = sdc_encrypt_decrypt_type_get_opt_bmsk(type, &opt_bmsk);

        /* using formatted with additional options is not allowed */
        if ((err == SDC_OK) && (opt_bmsk != 0))
            err = SDC_OP_NOT_SUPPORTED;
    }

    if (err == SDC_OK) {
        /* determine the output length */
        err = sdc_arch_encrypt_get_out_len(session, type, &internal_desc, in_len, &internal_out_len);
    }

    if (err == SDC_OK) {
        sdc_inter_form_header_init(&form_header);

        err = sdc_intern_encrypt_decrypt_formatted_fill_header (session,
                                                                type,
                                                                internal_out_len,
                                                                internal_iv_len,
                                                                &form_header);

        if (err == SDC_OK) {
            internal_formatted_buffer = malloc(form_header.overall_formatted_len);
            if (internal_formatted_buffer == NULL)
                err = SDC_NO_MEM;
        }

        if (err == SDC_OK) {
            err = sdc_intern_encrypt_decrypt_formatted_update_pointers(&form_header,
                                                                       internal_formatted_buffer);
        }

        if (err == SDC_OK) {
            form_encdec = &form_header.encrypt_decrypt;
            err = sdc_intern_formatted_write_header(&form_header,
                                                    internal_formatted_buffer);

            if (err == SDC_OK) {
                if (internal_iv_len != 0) {
                    if (explicit_iv) {
                        /* copy explicitly specified iv */
                        memcpy (form_encdec->iv, iv, iv_len);
                    } else {
                        iv_tmp = NULL;
                        err = sdc_random_gen_buffer(session, internal_iv_len, &iv_tmp);
                        if (err == SDC_OK) {
                            memcpy (form_encdec->iv, iv_tmp, internal_iv_len);
                        }
                        if (iv_tmp != NULL)
                            free(iv_tmp);
                    }
                }
            }

            if (err == SDC_OK) {
                err = sdc_encrypt_selector(session,
                                           type,
                                           &internal_desc,
                                           in_data, in_len,
                                           form_encdec->data, form_encdec->data_len,
                                           form_encdec->iv, form_encdec->iv_len);
            }
        }

        if (err == SDC_OK) {
            *formatted_data = internal_formatted_buffer;
            *formatted_len = form_header.overall_formatted_len;
        } else {
            if (internal_formatted_buffer)
                free (internal_formatted_buffer);
        }

        sdc_inter_form_header_free(&form_header);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_ENCRYPT, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_encrypt_formatted(sdc_session_t *session,
                                  const sdc_encrypt_decrypt_type_t *type,
                                  const uint8_t *in_data, const size_t in_len,
                                  uint8_t **formatted_data, size_t *formatted_len)
{
    return sdc_encrypt_formatted_extended(session,
                                          type,
                                          in_data, in_len,
                                          NULL, SDC_IV_USE_DEFAULT,
                                          formatted_data, formatted_len);
}

sdc_error_t sdc_decrypt_formatted(sdc_session_t *session,
                                  const sdc_encrypt_decrypt_type_t *type,
                                  const uint8_t *formatted_data, const size_t formatted_len,
                                  uint8_t **out_data, size_t *out_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;
    sdc_form_header_generic_t form_header;
    sdc_form_header_encrypt_decrypt_generic_t *form_encdec = NULL;
    sdc_encrypt_decrypt_desc_t desc;
    uint8_t *internal_out = NULL;
    size_t internal_out_len = 0;
    sdc_encrypt_decrypt_alg_t alg;
    sdc_encrypt_decrypt_blk_t blk;

    if (SDC_OK != sdc_intern_check_init_data_output_buffer(out_data, out_len))
        err = SDC_OUT_DATA_INVALID;

    if (SDC_OK != sdc_intern_check_data_input_buffer(formatted_data, formatted_len))
        err = SDC_FORMATTED_DATA_INVALID;

    if (!session)
        err = SDC_SESSION_INVALID;

    if (err != SDC_OK)
        return err;

    err = sdc_intern_formatted_read_header(
            &form_header,
            formatted_data, formatted_len);


    if (err == SDC_OK) {
        if (form_header.operation != SDC_FORMATTED_TYPE_ENCRYPT_DECRYPT)
            err = SDC_FORMATTED_DATA_INVALID;
    }

    if (err == SDC_OK) {
        form_encdec = &form_header.encrypt_decrypt;
        /* we need to typecast as the same pointers in encrypt struct are used for
         * read and write. Before calling decrypt we will make it const again */
        err = sdc_intern_encrypt_decrypt_formatted_update_pointers(
            &form_header,
            (uint8_t*)formatted_data);

        /* check if given type matches the one from the formatted data */
        if (err == SDC_OK) {
            err = sdc_encrypt_decrypt_type_get_algorithm(type, &alg);

            if ((err == SDC_OK) && (alg != form_encdec->alg))
                err = SDC_ALG_MODE_INVALID;
        }
        if (err == SDC_OK) {
            err = sdc_encrypt_decrypt_type_get_block_mode(type, &blk);

            if ((err == SDC_OK) && (blk != form_encdec->blk))
                err = SDC_ALG_MODE_INVALID;
        }

        if (err == SDC_OK) {
            uint64_t opt_bmsk;

            err = sdc_encrypt_decrypt_type_get_opt_bmsk(type, &opt_bmsk);

            /* using formatted with additional options is not allowed */
            if ((err == SDC_OK) && (opt_bmsk != 0))
                err = SDC_OP_NOT_SUPPORTED;
        }

        if (err == SDC_OK) {
            err = sdc_encrypt_decrypt_desc_fill (session, type, &desc);

            if (err == SDC_OK) {
                err = sdc_encrypt_decrypt_common_check_iv_in_length(&desc,
                                                                    false,
                                                                    form_encdec->data_len,
                                                                    form_encdec->iv_len);

                if (err == SDC_OK) {
                    /* determine the max output length */
                    err = sdc_arch_decrypt_get_max_out_len(
                        session, type, &desc,
                        form_encdec->data_len,
                        &internal_out_len);
                }


                if ((err == SDC_OK) && (internal_out_len != 0)) {
                    /* allocate output buffer */
                    internal_out = malloc(internal_out_len);
                    if (!internal_out)
                        err = SDC_NO_MEM;
                }

                if (err == SDC_OK) {
                    err = sdc_decrypt_selector(session,
                                               type,
                                               &desc,
                                               (const uint8_t*)form_encdec->data,
                                               form_encdec->data_len,
                                               internal_out,
                                               &internal_out_len,
                                               (const uint8_t*)form_encdec->iv, form_encdec->iv_len);
                }
            }
        }
    }

    if (err == SDC_OK) {
        *out_data = internal_out;
        *out_len = internal_out_len;
    } else {
        if (internal_out)
            free(internal_out);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_DECRYPT, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_decrypt_formatted_autoload_key_with_secret_mod(
    sdc_session_t *session,
    const uint8_t *formatted_data, const size_t formatted_len,
    const uint8_t *secret_mod_data, const size_t secret_mod_len)
{
    return sdc_intern_formatted_autoload_key_with_secret_mod(
               session,
               formatted_data, formatted_len,
               secret_mod_data, secret_mod_len);
}

sdc_error_t sdc_decrypt_formatted_autoload_key(sdc_session_t *session,
                                               const uint8_t *formatted_data, const size_t formatted_len)
{
    return sdc_unwrap_formatted_autoload_key_with_secret_mod(
               session,
               formatted_data, formatted_len,
               NULL, 0);
}

sdc_error_t sdc_decrypt_formatted_extract_type(
    const uint8_t *formatted_data, const size_t formatted_len,
    sdc_encrypt_decrypt_type_t **type)
{
    sdc_error_t err = SDC_OK;
    sdc_form_header_generic_t form_header;
    sdc_encrypt_decrypt_type_t *tmp = NULL;

    if (SDC_OK != sdc_intern_check_data_input_buffer(formatted_data, formatted_len))
        return SDC_FORMATTED_DATA_INVALID;

    if (!type) {
        return SDC_ALG_MODE_INVALID;
    }

    // initialize
    *type = NULL;

    err = sdc_intern_formatted_read_header (
        &form_header,
        formatted_data, formatted_len);

    if (err == SDC_OK) {
        if (form_header.operation != SDC_FORMATTED_TYPE_ENCRYPT_DECRYPT)
            err = SDC_FORMATTED_DATA_INVALID;
    }

    if (err == SDC_OK) {
        err = sdc_encrypt_decrypt_type_alloc(&tmp);
    }

    if (err == SDC_OK) {
        err = sdc_encrypt_decrypt_type_set_algorithm(tmp, form_header.encrypt_decrypt.alg);
    }

    if (err == SDC_OK) {
        err = sdc_encrypt_decrypt_type_set_block_mode(tmp, form_header.encrypt_decrypt.blk);
    }

    if (err == SDC_OK) {
        *type = tmp;
    } else {
        if (tmp != NULL)
            sdc_encrypt_decrypt_type_free(tmp);
    }

    return err;
}

/* defined in sdc_op_common.h */
const sdc_encrypt_decrypt_type_t *sdc_encrypt_decrypt_get_default(void)
{
    return sdc_arch_encrypt_decrypt_get_default();
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_type_alloc(sdc_encrypt_decrypt_type_t **type)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_encrypt_decrypt_type_alloc(type);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_type_free(sdc_encrypt_decrypt_type_t *type)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_encrypt_decrypt_type_free(type);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_type_set_algorithm(sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_alg_t alg)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if ((alg < SDC_ENCDEC_ALG_FIRST) || (alg >= SDC_ENCDEC_ALG_END))
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_encrypt_decrypt_type_set_algorithm(type, alg);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_type_set_block_mode(sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_blk_t blk)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if ((blk < SDC_ENCDEC_BLK_FIRST) || (blk >= SDC_ENCDEC_BLK_END))
        return SDC_ALG_MODE_INVALID;

    return sdc_arch_encrypt_decrypt_type_set_block_mode(type, blk);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_type_set_opt_bmsk(sdc_encrypt_decrypt_type_t *type, uint64_t opt_bmsk)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (opt_bmsk != 0) /* nothing supported so far */
        return SDC_OP_NOT_SUPPORTED;

    /* as we don't support any option at the moment, we don't need any arch interface */
    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_type_get_algorithm(const sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_alg_t *alg)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!alg)
        return SDC_INVALID_PARAMETER;

    return sdc_arch_encrypt_decrypt_type_get_algorithm(type, alg);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_type_get_block_mode(const sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_blk_t *blk)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!blk)
        return SDC_INVALID_PARAMETER;

    return sdc_arch_encrypt_decrypt_type_get_block_mode(type, blk);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_type_get_opt_bmsk(const sdc_encrypt_decrypt_type_t *type, uint64_t *opt_bmsk)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!opt_bmsk)
        return SDC_INVALID_PARAMETER;

    /* as we don't support any option at the moment, we don't need any arch interface - simply return 0*/
    *opt_bmsk = 0;
    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_desc_alloc(sdc_encrypt_decrypt_desc_t **desc)
{
    if (desc == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    *desc = malloc(sizeof(sdc_encrypt_decrypt_desc_t));
    if (*desc == NULL) {
        return SDC_NO_MEM;
    }

    memset(*desc, 0, sizeof(sdc_encrypt_decrypt_desc_t));

    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_desc_free(sdc_encrypt_decrypt_desc_t *desc)
{
    if (desc == NULL) {
        return SDC_INVALID_PARAMETER;
    }

    free(desc);

    return SDC_OK;
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_desc_fill (
    sdc_session_t *session,
    const sdc_encrypt_decrypt_type_t *type,
    sdc_encrypt_decrypt_desc_t *desc)
{
    if (!type)
        return SDC_ALG_MODE_INVALID;

    if (!desc)
        return SDC_INVALID_PARAMETER;

    return sdc_arch_encrypt_decrypt_desc_fill(session,type, desc);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_desc_get_iv(sdc_encrypt_decrypt_desc_t *desc,
                                            size_t *min_val,
                                            size_t *max_val,
                                            size_t *mod,
                                            size_t *default_val)
{
    if (!desc)
        return SDC_INVALID_PARAMETER;

    return sdc_intern_range_get_min_max_mod_dflt(&(desc->iv),
                                           min_val,
                                           max_val,
                                           mod,
                                           default_val);
}

/* defined in sdc_op_common.h */
sdc_error_t sdc_encrypt_decrypt_desc_get_max_chunk_len(const sdc_encrypt_decrypt_desc_t *desc,
                                                       size_t *max_val)
{
    size_t max_chunk;

    if (!desc)
        return SDC_INVALID_PARAMETER;

    if (max_val) {
        max_chunk = desc->data.max_chunk_len;
        if (desc->data.chunk_len_aligned) {
            max_chunk -= (max_chunk % desc->data.block_len);
        }

        if (max_chunk == 0)
            return SDC_INTERNAL_ERROR;

        *max_val = max_chunk;
    }

    return SDC_OK;
}

const char* sdc_encrypt_decrypt_algorithm_name (sdc_encrypt_decrypt_alg_t alg)
{
    size_t idx;
    size_t elems;

    if ((alg < SDC_ENCDEC_ALG_FIRST) || (alg >= SDC_ENCDEC_ALG_END))
        return NULL;

    idx = alg;
    elems = sizeof(sdc_encrypt_decrypt_alg_names) / sizeof(const char *);

    if (idx<elems) {
        return sdc_encrypt_decrypt_alg_names[idx];
    }

    return NULL;
}

const char* sdc_encrypt_decrypt_block_mode_name (sdc_encrypt_decrypt_blk_t blk)
{
    size_t idx;
    size_t elems;

    if ((blk < SDC_ENCDEC_BLK_FIRST) || (blk >= SDC_ENCDEC_BLK_END))
        return NULL;

    idx = blk;
    elems = sizeof(sdc_encrypt_decrypt_blk_names) / sizeof(const char *);

    if (idx<elems) {
        return sdc_encrypt_decrypt_blk_names[idx];
    }

    return NULL;
}


sdc_error_t sdc_encrypt_decrypt_get_key_key_lens_fmt(
    sdc_encrypt_decrypt_type_t *type,
    sdc_key_fmt_t *sup_key_fmt_protect,
    sdc_key_fmt_t *sup_key_fmt_unprotect,
    sdc_key_len_bmsk_t *sup_key_lens,
    sdc_key_len_t *dflt_key_len)
{
    sdc_key_desc_t key_desc;
    sdc_error_t err;

    if (!type)
        return SDC_ALG_MODE_INVALID;

    err = sdc_arch_encrypt_decrypt_key_desc_fill(type, &key_desc);

    if (err == SDC_OK)
        err = sdc_intern_key_lens_fmt(&key_desc,
                                      sup_key_fmt_protect, sup_key_fmt_unprotect,
                                      sup_key_lens, dflt_key_len);

    return err;
}

static sdc_error_t sdc_session_type_desc_validation(sdc_session_t *session,
                                              const sdc_encrypt_decrypt_type_t *type,
                                              const sdc_encrypt_decrypt_desc_t *desc)
{
    /* verify inputs */
    if (!session)
        return SDC_SESSION_INVALID;
    if(!type)
        return SDC_ALG_MODE_INVALID;
    if(!desc)
        return SDC_INVALID_PARAMETER;

    return SDC_OK;
}

/**
 * \brief call min max checks for in data and iv
 */
static sdc_error_t sdc_encrypt_decrypt_common_check_iv(const sdc_encrypt_decrypt_desc_t *desc, size_t iv_len )
{
    if (sdc_intern_range_min_max_mod_check(iv_len, &(desc->iv)) != SDC_OK)
        return SDC_IV_INVALID;

    return SDC_OK;
}

/* check output parameters */
static sdc_error_t sdc_encrypt_decrypt_update_check_data_output (const uint8_t *data, size_t *len)
{
    sdc_error_t err = SDC_OK;

    if ((data == NULL) || (len == NULL))
        err = SDC_OUT_DATA_INVALID;

    return err;
}

sdc_error_t sdc_encrypt_get_overall_len(sdc_session_t *session,
                                        const sdc_encrypt_decrypt_type_t *type,
                                        const sdc_encrypt_decrypt_desc_t *desc,
                                        const size_t in_len, size_t *out_len_max)
{
    sdc_error_t err;

    if(out_len_max == NULL)
        return SDC_OUT_DATA_INVALID;

    err = sdc_session_type_desc_validation(session, type, desc);
    if(err != SDC_OK)
         return err;

    /* in fact it will be the exact length */
    return sdc_arch_encrypt_get_out_len(session, type, desc, in_len, out_len_max);
}


sdc_error_t sdc_decrypt_get_overall_len(sdc_session_t *session,
                                        const sdc_encrypt_decrypt_type_t *type,
                                        const sdc_encrypt_decrypt_desc_t *desc,
                                        const size_t in_len, size_t *out_len_max)
{
    sdc_error_t err;

    if(out_len_max == NULL)
        return SDC_OUT_DATA_INVALID;

    err = sdc_session_type_desc_validation(session, type, desc);
    if(err != SDC_OK)
         return err;

    return sdc_arch_decrypt_get_max_out_len(session, type, desc, in_len, out_len_max);
}

sdc_error_t sdc_encrypt_get_update_len(sdc_session_t *session,
                                       const sdc_encrypt_decrypt_type_t *type,
                                       const sdc_encrypt_decrypt_desc_t *desc,
                                       const size_t in_len, size_t *out_len_max)
{
    sdc_error_t err = SDC_OK;
    sdc_padding_t padding;
    size_t effective_in_length;

    if(out_len_max == NULL)
        err = SDC_OUT_DATA_INVALID;

    if (err == SDC_OK)
        err = sdc_session_type_desc_validation(session, type, desc);

    if (err == SDC_OK) {
        padding = desc->data.padding;

        effective_in_length = in_len;
        if (desc->data.chunk_len_aligned) {
            /*
             * when chunk_len_aligned only block_len aligned length are
             * forwarded to the arch code
             */
            effective_in_length -= effective_in_length % desc->data.block_len;
            /* but unaligned buffer may contain one block */
            effective_in_length += desc->data.block_len;
        }

        /*
         * in case of padding at least for finalize one block size (padding)
         * may be passed to the arch code
         */
        if ((padding != SDC_PADDING_INTERNAL) &&
            (padding != SDC_PADDING_NO) &&
            (padding != SDC_PADDING_HIDDEN) &&
            (effective_in_length < desc->data.block_len)) {
            effective_in_length = desc->data.block_len;
        }

        err = sdc_arch_encrypt_get_update_len(session,
                                              type, desc,
                                              effective_in_length,
                                              out_len_max);
    }

    return err;
}

sdc_error_t sdc_decrypt_get_update_len(sdc_session_t *session,
                                       const sdc_encrypt_decrypt_type_t *type,
                                       const sdc_encrypt_decrypt_desc_t *desc,
                                       const size_t in_len, size_t *out_len_max)
{
    sdc_error_t err = SDC_OK;
    sdc_padding_t padding;
    bool unaligned_buffer_used = false;
    size_t effective_in_length;

    if(out_len_max == NULL)
        err = SDC_OUT_DATA_INVALID;

    if (err == SDC_OK)
        err = sdc_session_type_desc_validation(session, type, desc);

    if (err == SDC_OK) {
        padding = desc->data.padding;

        /* unaligned buffer will hold back up to one block */
        if ((padding != SDC_PADDING_INTERNAL) &&
            (padding != SDC_PADDING_NO) &&
            (padding != SDC_PADDING_HIDDEN))
            unaligned_buffer_used = true;

        effective_in_length = in_len;
        if (desc->data.chunk_len_aligned) {
            /*
             * when chunk_len_aligned only block_len aligned length are
             * forwarded to the arch code
             */
            effective_in_length -= effective_in_length % desc->data.block_len;
            /* but unaligned buffer may contain one block */
            unaligned_buffer_used = true;
        }

        if (unaligned_buffer_used)
            effective_in_length += desc->data.block_len;

        err = sdc_arch_decrypt_get_update_len(session,
                                              type, desc,
                                              effective_in_length,
                                              out_len_max);
    }

    return err;
}

sdc_error_t sdc_encrypt_init(sdc_session_t *session,
                             const sdc_encrypt_decrypt_type_t *type,
                             const sdc_encrypt_decrypt_desc_t *desc,
                             uint8_t **iv, size_t *iv_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;

    bool explicit_iv = false;
    uint8_t *internal_iv = NULL;
    size_t internal_iv_len = 0;

    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_ENCRYPT, SDC_NO_OP);

    /* initialize - in case failing with error we want to return
     * buffer pointer NULL and buffer len 0 for all not explicitly specified buffers
     */
    if (err == SDC_OK) {
        if (sdc_intern_check_init_dgst_tag_iv_output_buffer(iv, iv_len, &internal_iv_len, SDC_IV_USE_DEFAULT, &explicit_iv) != SDC_OK) {
            err = SDC_IV_INVALID;
        }
    }

    if (err == SDC_OK) {
        if (!explicit_iv) {
            if (internal_iv_len == SDC_IV_USE_DEFAULT) {
                internal_iv_len = desc->iv.dflt;
            }
            internal_iv = NULL;
            /* there might be formats without IV */
            if (internal_iv_len != 0) {
                err = sdc_random_gen_buffer(session, internal_iv_len, &internal_iv);
            }
        } else {
            internal_iv = *iv;
            internal_iv_len = *iv_len;
        }
    }

    if (err == SDC_OK) {
        err = sdc_encrypt_decrypt_common_check_iv(desc, internal_iv_len);
    }

    if (err == SDC_OK) {
        err = sdc_common_encdec_init(session, type, desc,
                                     internal_iv, internal_iv_len,
                                     true);
    }

    if (err == SDC_OK) {
        if (!explicit_iv) {
            *iv = internal_iv;
            *iv_len = internal_iv_len;
        }
    } else {
        /* clean allocated memory */
        if (!explicit_iv)
            free (internal_iv);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_ENCRYPT, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_ENCRYPT, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_decrypt_init(sdc_session_t *session,
                             const sdc_encrypt_decrypt_type_t *type,
                             const sdc_encrypt_decrypt_desc_t *desc,
                             const uint8_t *iv, size_t iv_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_DECRYPT, SDC_NO_OP);

    if(err == SDC_OK) {
        if (sdc_intern_check_tag_iv_input_buffer(iv, iv_len, SDC_IV_USE_DEFAULT) != SDC_OK) {
            err = SDC_IV_INVALID;
        }
    }

    if (err == SDC_OK) {
        err = sdc_encrypt_decrypt_common_check_iv(desc, iv_len);
    }

    if (err == SDC_OK) {
        err = sdc_common_encdec_init(session, type, desc,
                                     iv, iv_len,
                                     false);
    }

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_DECRYPT, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_DECRYPT, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_encrypt_update(sdc_session_t *session,
                               const sdc_encrypt_decrypt_type_t *type,
                               const sdc_encrypt_decrypt_desc_t *desc,
                               const uint8_t *in_data, const size_t in_len,
                               uint8_t *out_data, size_t *out_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_ENCRYPT, SDC_OP_INITILIZED);

    if(err == SDC_OK)
        err = sdc_encrypt_decrypt_update_check_data_output(out_data, out_len);

    if (err == SDC_OK) {
        if (sdc_intern_check_data_input_buffer(in_data, in_len) != SDC_OK)
            err = SDC_IN_DATA_INVALID;
    }

    if(err == SDC_OK)
        err = sdc_common_encdec_update(session, type, desc,
                                       in_data, in_len,
                                       out_data, out_len,
                                       true);

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_ENCRYPT, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_ENCRYPT, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_decrypt_update(sdc_session_t *session,
                               const sdc_encrypt_decrypt_type_t *type,
                               const sdc_encrypt_decrypt_desc_t *desc,
                               const uint8_t *in_data, const size_t in_len,
                               uint8_t *out_data, size_t *out_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;

    /* verify inputs */
    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_DECRYPT, SDC_OP_INITILIZED);

    if(err == SDC_OK)
        err = sdc_encrypt_decrypt_update_check_data_output(out_data, out_len);

    if (err == SDC_OK) {
        if (sdc_intern_check_data_input_buffer(in_data, in_len) != SDC_OK)
        err = SDC_IN_DATA_INVALID;
    }

    if(err == SDC_OK)
        err = sdc_common_encdec_update(session, type, desc,
                                       in_data, in_len,
                                       out_data, out_len,
                                       false);

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_DECRYPT, SDC_OP_INITILIZED);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_DECRYPT, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_encrypt_finalize(sdc_session_t *session,
                                 const sdc_encrypt_decrypt_type_t *type,
                                 const sdc_encrypt_decrypt_desc_t *desc,
                                 uint8_t *out_data, size_t *out_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;

    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_ENCRYPT, SDC_OP_INITILIZED);

    if(err == SDC_OK)
        err = sdc_encrypt_decrypt_update_check_data_output(out_data, out_len);

    if (err == SDC_OK)
        err = sdc_common_encdec_finalize(session, type, desc,
                                         out_data, out_len,
                                         true);

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_ENCRYPT, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}

sdc_error_t sdc_decrypt_finalize(sdc_session_t *session,
                                 const sdc_encrypt_decrypt_type_t *type,
                                 const sdc_encrypt_decrypt_desc_t *desc,
                                 uint8_t *out_data, size_t *out_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err = SDC_OK;

    err = sdc_session_type_desc_validation(session, type, desc);

    if(err == SDC_OK)
        err = sdc_intern_ops_sequence_ctl_check(session, SDC_OP_DECRYPT, SDC_OP_INITILIZED);

    if(err == SDC_OK)
        err = sdc_encrypt_decrypt_update_check_data_output(out_data, out_len);

    if (err == SDC_OK)
        err = sdc_common_encdec_finalize(session, type, desc,
                                         out_data, out_len,
                                         false);

    if(err == SDC_OK) {
        err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_NO_CRYPTO, SDC_NO_OP);
    } else {
        /* If current operation is failed, reset the history of "operation" and "function" parameters
         * since doesn't support retry operation */
        tmp_err = sdc_intern_ops_sequence_ctl_set(session, SDC_OP_DECRYPT, SDC_OP_ERROR);
        if(tmp_err != SDC_OK)
            err = tmp_err;
    }

    return err;
}
